# AlexShop Deployment Guide

## Project Structure

```
alexshopclone/
├── backend/                 # Node.js backend services
│   ├── websocket_server.js  # WebSocket server (port 3001)
│   ├── api_server.js       # API server (port 3002)
│   ├── start.js            # Startup script
│   ├── config.js           # Configuration
│   ├── package.json        # Dependencies
│   └── node_modules/       # Node.js modules
├── config/
│   └── backend.php         # PHP backend configuration
├── admin/                  # Admin dashboard
├── database/               # JSON data files
├── functions/              # PHP functions
├── includes/               # PHP includes
└── *.php                   # Main PHP files
```

## Local Development Setup

### 1. Start Backend Services
```bash
# Windows
start_backend.bat

# Linux/Mac
chmod +x start_backend.sh
./start_backend.sh
```

### 2. Access Application
- **Main Site**: `http://localhost/alexshopclone/`
- **Admin Dashboard**: `http://localhost/alexshopclone/admin/`
- **WebSocket**: `ws://localhost:3001/ws`
- **API**: `http://localhost:3002/api/visitors`

## VPS Deployment

### 1. Upload Files
Upload the entire project to your VPS:
```bash
# Example using SCP
scp -r alexshopclone/ user@your-vps-ip:/var/www/html/
```

### 2. Configure Backend URLs
Edit `config/backend.php` and update the URLs:
```php
// Replace with your VPS domain/IP
define('WEBSOCKET_URL', 'ws://your-domain.com:3001/ws');
define('API_URL', 'http://your-domain.com:3002/api/visitors');
define('SEND_REQUEST_URL', 'http://your-domain.com:3001/api/send-request');
```

### 3. Install Node.js on VPS
```bash
# Install Node.js (Ubuntu/Debian)
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt-get install -y nodejs

# Or use Node Version Manager
curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.39.0/install.sh | bash
nvm install 18
nvm use 18
```

### 4. Install Backend Dependencies
```bash
cd /var/www/html/alexshopclone/backend
npm install --production
```

### 5. Start Backend Services
```bash
# Using PM2 (recommended for production)
npm install -g pm2
pm2 start start.js --name "alexshop-backend"
pm2 save
pm2 startup

# Or using systemd service
sudo systemctl start alexshop-backend
sudo systemctl enable alexshop-backend
```

### 6. Configure Firewall
```bash
# Allow ports 3001 and 3002
sudo ufw allow 3001
sudo ufw allow 3002
sudo ufw allow 80
sudo ufw allow 443
```

### 7. Configure Nginx (Optional)
```nginx
# /etc/nginx/sites-available/alexshop
server {
    listen 80;
    server_name your-domain.com;
    
    # PHP files
    location / {
        root /var/www/html/alexshopclone;
        index index.php;
        try_files $uri $uri/ /index.php?$query_string;
    }
    
    # Proxy WebSocket
    location /ws {
        proxy_pass http://localhost:3001;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection "upgrade";
        proxy_set_header Host $host;
    }
    
    # Proxy API
    location /api/ {
        proxy_pass http://localhost:3002;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
    }
}
```

## Environment Configuration

### Development
- Backend URLs automatically detect localhost
- Debug mode enabled
- CORS allows all origins

### Production
- Manual URL configuration required
- Debug mode disabled
- Configure proper CORS settings
- Use HTTPS for WebSocket connections

## Troubleshooting

### Backend Services Not Starting
```bash
# Check if ports are in use
netstat -tulpn | grep :3001
netstat -tulpn | grep :3002

# Check Node.js processes
ps aux | grep node
```

### PHP Can't Connect to Backend
1. Check if backend services are running
2. Verify firewall settings
3. Check URL configuration in `config/backend.php`
4. Test API endpoints manually

### WebSocket Connection Issues
1. Ensure WebSocket server is running on port 3001
2. Check browser console for connection errors
3. Verify CORS settings
4. Test with WebSocket client tools

## Monitoring

### PM2 Monitoring
```bash
pm2 status
pm2 logs alexshop-backend
pm2 monit
```

### System Monitoring
```bash
# Check service status
systemctl status alexshop-backend

# View logs
journalctl -u alexshop-backend -f
```

## Security Considerations

1. **Firewall**: Only open necessary ports
2. **HTTPS**: Use SSL certificates for production
3. **CORS**: Configure proper CORS settings
4. **Authentication**: Ensure admin pages are protected
5. **File Permissions**: Set proper file permissions
6. **Updates**: Keep Node.js and dependencies updated
